/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: OS_SPICustom.c 12222 2018-07-26 07:03:23Z Robert $:

  Description:
    Implementation of the custom SPI abstration layer

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-07-26  Added return value to OS_SpiInit() and changed compile #error to
                #warning
    2014-08-27  created

**************************************************************************************/

/*****************************************************************************/
/*! \file OS_SPICustom.c
*    Sample SPI abstraction layer. Implementation must be done
*    according to used target system                                         */
/*****************************************************************************/

#include "OS_Spi.h"
#include "ToolkitSample.h"
#include <hal_resources_defines_netx51.h>

#ifdef CIFX_TOOLKIT_HWIF
  #warning "Implement SPI target system abstraction in this file"
#endif

/*****************************************************************************/
/*!  \addtogroup CIFX_TK_OS_ABSTRACTION Operating System Abstraction
*    \{                                                                      */
/*****************************************************************************/
#define SERDPM_SLAVE 1
static NX51_SPI_AREA_T* s_ptSpi  = (NX51_SPI_AREA_T*) NX51_NETX_SPI1_AREA;

/*****************************************************************************/
/*! Initialize SPI components
*   \param pvOSDependent OS Dependent parameter
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
long OS_SpiInit(void* pvOSDependent)
{
  /* initialize SPI device */

  /* setup SPI Core data width and enable static nCS handling */
  s_ptSpi->aulSpi_cr[0]  = 7 << SRT_NX51_spi_cr0_datasize;

  /* clock divider for SCK  to achieve 12,5MHz*/
  s_ptSpi->aulSpi_cr[0] |= ((12500 * 4096 / 100000) << SRT_NX51_spi_cr0_sck_muladd) & MSK_NX51_spi_cr0_sck_muladd;

  /* SPI Mode 3 */
  s_ptSpi->aulSpi_cr[0] |= MSK_NX51_spi_cr0_SPH | MSK_NX51_spi_cr0_SPO;
  /* Clear fifos, set cs mode */
  s_ptSpi->aulSpi_cr[1] = MSK_NX51_spi_cr1_rx_fifo_clr | (8 << SRT_NX51_spi_cr1_rx_fifo_wm) |
                          MSK_NX51_spi_cr1_tx_fifo_clr | (8 << SRT_NX51_spi_cr1_tx_fifo_wm) |
                          MSK_NX51_spi_cr1_fss_static  | MSK_NX51_spi_cr1_SSE;

  return 0;
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  /* Clear tx/rx FIFO */
  s_ptSpi->aulSpi_cr[1] |= (MSK_NX51_spi_cr1_rx_fifo_clr | MSK_NX51_spi_cr1_tx_fifo_clr );
  /* assert chip select */
  s_ptSpi->aulSpi_cr[1] |= (1 << (SRT_NX51_spi_cr1_fss+SERDPM_SLAVE));
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  /* deassert chip select */
  s_ptSpi->aulSpi_cr[1] &= ~(1 << (SRT_NX51_spi_cr1_fss+SERDPM_SLAVE));
}

/*****************************************************************************/
/*! Lock the SPI bus
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiLock(void* pvOSDependent)
{
  /* lock access to SPI device */
  NONOS_DEVICEINSTANCE_T* ptInst = (NONOS_DEVICEINSTANCE_T*)pvOSDependent;
  lock_irqfiq_save(ptInst->ulSerLock);    /*lint !e522 !e529 !e160 : function call lock_irqfiq_save() */
}  /*lint !e438 !e529 */

/*****************************************************************************/
/*! Unlock the SPI bus
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiUnlock(void* pvOSDependent)
{
  /* unlock access to SPI device */
  NONOS_DEVICEINSTANCE_T* ptInst = (NONOS_DEVICEINSTANCE_T*)pvOSDependent;
  lock_irqfiq_restore(ptInst->ulSerLock); /*lint !e522 !e529 !e160 : function call lock_irqfiq_restore() */
}  /*lint !e438 !e529 */

/*****************************************************************************/
/*! Transfer byte stream via SPI
*   \param pvOSDependent OS Dependent parameter to identify card
*   \param pbSend        Send buffer (NULL for polling)
*   \param pbRecv        Receive buffer (NULL if discard)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  uint32_t ulTxLen  = ulLen;

  while(ulLen != 0)
  {
    if ((ulTxLen) && (s_ptSpi->ulSpi_sr&MSK_NX51_spi_sr_TNF) && !(s_ptSpi->ulSpi_sr & MSK_NX51_spi_sr_RFF))
    {
      --ulTxLen;
      if (pbSend == NULL)
        s_ptSpi->ulSpi_dr = 0x00;
      else
        s_ptSpi->ulSpi_dr = *(pbSend++);
    }
    if(s_ptSpi->ulSpi_sr & MSK_NX51_spi_sr_RNE)
    {
      --ulLen;
      if (pbRecv == NULL)
        s_ptSpi->ulSpi_dr;
      else
        *(pbRecv++) = s_ptSpi->ulSpi_dr;
    }
  }
}
/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
